from fastapi.encoders import jsonable_encoder
from fastapi.responses import JSONResponse
from logger import logger

# controllers
from controller.context_manager import context_user_data

# schema
from schema.base import GenericResponseModel

from typing import List
from pydantic import ValidationError


# build a proper api response from the Generic response sent to it
def build_api_response(generic_response: GenericResponseModel) -> JSONResponse:
    try:
        response_json = jsonable_encoder(generic_response)

        # Remove the status_code key if it exists
        response_json.pop("status_code", None)

        res = JSONResponse(
            status_code=generic_response.status_code, content=response_json
        )

        # logger.info(
        #     extra=context_user_data.get(),
        #     msg="build_api_response: Generated Response with status_code:"
        #     + f"{generic_response.status_code}",
        # )
        return res

    except Exception as e:
        logger.error(
            extra=context_user_data.get(),
            msg=f"exception in build_api_response error : {e}",
        )

        return JSONResponse(status_code=generic_response.status_code, content=str(e))


def format_validation_errors(errors: List[dict]) -> dict:

    error = errors[0]
    message = error["msg"]
    fields_string = ", ".join(f"{field}" for field in error["loc"][1:])

    return {
        "data": {"fields": fields_string},
        "message": message,
        "status": False,
    }


def handle_validation_error(exc: ValidationError) -> JSONResponse:
    return JSONResponse(status_code=422, content=format_validation_errors(exc.errors()))
