import http
import json
from psycopg2 import DatabaseError
import requests
import platform

from controller.context_manager import context_user_data
from pydantic import BaseModel
from model.user import User
import traceback

from schema.base import GenericResponseModel
from schema.user import (
    UserInsertModel,
    UserLoginModel,
    UserModel,
    UserTokenResponseModel,
)

from datetime import datetime, timedelta

from utils.jwt_token_handler import JWTHandler
from utils.password_hasher import PasswordHasher

from logger import logger


class AccessKeyRequestModel(BaseModel):
    access_key: str


class UserService:
    MSG_USER_CREATED_SUCCESS = "User created successfully"
    MSG_USER_LOGIN_SUCCESS = "Login successful"
    MSG_USER_SUSPENDED = "User is suspended successfully"

    ERROR_INVALID_CREDENTIALS = "Invalid credentials"
    ERROR_USER_NOT_FOUND = "User not found"

    @staticmethod
    def signup_user(
        user: UserInsertModel,
    ) -> GenericResponseModel:
        try:
            hashed_password = PasswordHasher.get_password_hash(user.password)

            # format the user and get dump
            formatted_user = user.create_db_entity(password_hash=hashed_password)

            # add user to database
            User.create_user(formatted_user)

            logger.info(
                extra=context_user_data.get(),
                msg="User created successfully with uuid {}".format(
                    formatted_user.uuid
                ),
            )

            return GenericResponseModel(
                status_code=http.HTTPStatus.CREATED,
                message=UserService.MSG_USER_CREATED_SUCCESS,
            )

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error creating user: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message="An error occurred while creating the user.",
            )

    @staticmethod
    def login_user(
        user_login_request: UserLoginModel,
    ) -> GenericResponseModel:

        user: UserModel = User.get_active_user_by_email(user_login_request.email)

        if not user:
            logger.error(
                extra=context_user_data.get(),
                msg="User not found",
            )

            return GenericResponseModel(
                status_code=http.HTTPStatus.NOT_FOUND,
                message="Invalid credentials",
            )

        if not PasswordHasher.verify_password(
            user_login_request.password, user.password_hash
        ):

            logger.error(
                extra=context_user_data.get(),
                msg="Invalid credentials",
            )
            return GenericResponseModel(
                status_code=http.HTTPStatus.UNAUTHORIZED,
                message=UserService.ERROR_INVALID_CREDENTIALS,
            )

        token = JWTHandler.create_access_token(json.loads(user.model_dump_json()))

        logger.info(
            extra=context_user_data.get(),
            msg=f"Login successful for user {user.email}" f" with token {token}",
        )

        user_data = json.loads(user.model_dump_json())

        if user_data["company_id"] is not None:

            # Safely remove the 'company_id' key if it exists
            user_data.pop("company_id", None)

        user_data_json = json.dumps(user_data)

        #  return token to client for further use
        return GenericResponseModel(
            status_code=http.HTTPStatus.OK,
            status=True,
            data=UserTokenResponseModel(
                access_token=token, user_data=json.loads(user_data_json)
            ),
            message=UserService.MSG_USER_LOGIN_SUCCESS,
        )

    @staticmethod
    def generate_access_token(
        access_key_request: AccessKeyRequestModel,
    ) -> GenericResponseModel:

        ACCESS_KEY = "b9c6ef40-9ef8-4bd6-ae1b-a9aa8e96fe40"
        ERROR_INVALID_ACCESS_KEY = "Invalid Access Key"

        if access_key_request.access_key != ACCESS_KEY:
            logger.error(
                extra=context_user_data.get(),
                msg="Invalid access key provided",
            )
            return GenericResponseModel(
                status_code=http.HTTPStatus.UNAUTHORIZED,
                message=ERROR_INVALID_ACCESS_KEY,
            )

        token = JWTHandler.create_access_token(
            to_encode={"access_key": access_key_request.access_key},
            expires_delta=timedelta(minutes=30),
        )

        # logger.info(
        #     extra=context_user_data.get(),
        #     msg=f"Access token generated successfully with token {token}",
        # )

        return GenericResponseModel(
            status_code=http.HTTPStatus.OK,
            status=True,
            data={"access_token": token},
            message="Token generated successfully",
        )

    def gati_test() -> GenericResponseModel:
        api_url = "https://justi.gati.com/webservices/GKEJCustVendDtls.jsp"

        headers = {
            "Content-Type": "application/json",
        }

        body = {
            "custCode": "30457301",
            "details": [
                {
                    "custVendorCode": "452316",
                    "custVendorName": "abcde",
                    "vendorAdd1": "vendorAdd1",
                    "vendorAdd2": "vendorAdd2",
                    "vendorAdd3": "vendorAdd3",
                    "vendorCity": "vendorCity",
                    "vendorPhoneNo": "vendorPhoneNo",
                    "vendorPincode": "110025",
                    "vendorEmail": "vendorEmail",
                    "vendorReceiverFlag": "V",
                    "vendorTinno": "3182738439dfdf7",
                    "VendorGSTNO": "1234567890",
                }
            ],
        }

        try:

            response = requests.post(api_url, data=body, headers=headers, verify=True)

            print(response.request)

            response.raise_for_status()  # Raises an HTTPError for bad responses

            response_data = response.json()

            return GenericResponseModel(
                status_code=http.HTTPStatus.OK,
                status=True,
                data=response_data,
                message="Successful",
            )

        except requests.exceptions.HTTPError as http_err:
            # Handle specific HTTP errors (e.g., 404, 500)
            error_message = f"HTTP error occurred: {str(http_err)}"
            return GenericResponseModel(
                status_code=response.status_code,
                status=False,
                data=None,
                message=error_message,
            )

        except requests.exceptions.ConnectionError as conn_err:
            # Handle connection-related errors
            error_message = f"Connection error occurred: {str(conn_err)}"
            return GenericResponseModel(
                status_code=http.HTTPStatus.SERVICE_UNAVAILABLE,
                status=False,
                data=None,
                message=error_message,
            )

        except requests.exceptions.Timeout as timeout_err:
            # Handle timeout errors
            error_message = f"Timeout error occurred: {str(timeout_err)}"
            return GenericResponseModel(
                status_code=http.HTTPStatus.REQUEST_TIMEOUT,
                status=False,
                data=None,
                message=error_message,
            )

        except requests.exceptions.RequestException as req_err:
            # Handle any other request-related errors
            error_message = f"Request error occurred: {str(req_err)}"
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                status=False,
                data=None,
                message=error_message,
            )

        except ValueError as val_err:
            # Handle JSON decoding errors
            error_message = f"JSON decoding error occurred: {str(val_err)}"
            return GenericResponseModel(
                status_code=http.HTTPStatus.BAD_REQUEST,
                status=False,
                data=None,
                message=error_message,
            )

        except Exception as e:
            # Handle any other unexpected errors
            error_message = f"An unexpected error occurred: {traceback.format_exc()}"
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                status=False,
                data=None,
                message=error_message,
            )
