import http
import json
import datetime
from psycopg2 import DatabaseError
import requests

from controller.context_manager import context_user_data
from controller.context_manager import get_db_session

from model.order import Order
from schema.base import GenericResponseModel
from schema.shipment import Shipment_create_model

from utils.jwt_token_handler import JWTHandler
from utils.password_hasher import PasswordHasher

from logger import logger

from couriers.max_logistics import Max_Logistics
from couriers.shipperfecto import Shipperfecto
from couriers.shiprocket import ShipRocket
from couriers.gati import Gati


class ShipmentService:

    @staticmethod
    def create_new_shipment(
        shipment_details: Shipment_create_model, courier: str
    ) -> GenericResponseModel:

        try:

            if courier == "maxlogistics":
                return Max_Logistics.createOrder(shipment_details)

            elif courier == "shipperfecto":
                return Shipperfecto.createOrder(shipment_details)

            elif courier == "shiprocket":
                return ShipRocket.createOrder(shipment_details)

            elif courier == "gati":
                return Gati.createOrder(shipment_details)

            else:
                return GenericResponseModel(
                    status_code=http.HTTPStatus.NOT_FOUND,
                    data={"courier_partner": courier},
                    message="Invalid Courier Partner",
                )

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error creating shipment: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="An error occurred while creating the shipment.",
            )

    @staticmethod
    def track_shipment(awb_number: str) -> GenericResponseModel:

        try:

            api_url = "https://apiv2.shiprocket.in/v1/external/auth/login"
            body = {
                "email": "moksh.jaswal1@omneelab.com",
                "password": "Moksh@Omnee@2024",
            }

            response = requests.post(api_url, json=body)
            response.raise_for_status()  # Raise an exception for HTTP errors
            response_data = response.json()

            if not response_data["token"]:
                return GenericResponseModel(
                    status_code=http.HTTPStatus.BAD_REQUEST,
                    message="Invalid shiprocket credentials",
                )

            token = response_data["token"]

            headers = {
                "Authorization": "Bearer " + token,
                "Content-Type": "application/json",
            }

            api_url = (
                "https://apiv2.shiprocket.in/v1/external/courier/track/awb/"
                + awb_number
            )

            response = requests.get(api_url, headers=headers, verify=False)
            response_data = response.json()

            tracking_data = response_data.get("tracking_data", "")

            if not tracking_data:
                return GenericResponseModel(
                    status_code=http.HTTPStatus.BAD_REQUEST,
                    message="Some error occurred, Please try again",
                )

            track_status = tracking_data.get("track_status", "")

            if track_status == 0:
                return GenericResponseModel(
                    status_code=http.HTTPStatus.BAD_REQUEST,
                    message=tracking_data.get(
                        "error", "Some error occured while tracking the shipment"
                    ),
                )

            current_status = tracking_data["shipment_track"][0]["current_status"]
            updated_awb_number = tracking_data["shipment_track"][0]["awb_code"]

            activites = tracking_data["shipment_track_activities"]

            with get_db_session() as db:
                order = db.query(Order).filter(Order.awb_number == awb_number).first()

                order.status = current_status
                order.awb_number = updated_awb_number

                old_tracking_info = order.tracking_info[:2]

                if activites:
                    new_tracking_info = [
                        {
                            "event": activity.get("sr-status-label", ""),
                            "subinfo": activity.get("activity", ""),
                            "date": activity.get("date", ""),
                            "location": activity.get("location", ""),
                        }
                        for activity in activites
                    ]

                    new_tracking_info.reverse()

                    track = old_tracking_info + new_tracking_info
                    order.tracking_info = track

                db.add(order)
                db.commit()

            return GenericResponseModel(
                status_code=http.HTTPStatus.OK,
                status=True,
                data={
                    "awb_number": updated_awb_number,
                    "current_status": current_status,
                },
                message="Tracking successfull",
            )

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error creating shipment: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="Some error occurred",
            )

    @staticmethod
    def track_all(self) -> GenericResponseModel:
        try:
            with get_db_session() as db:
                orders = db.query(Order).all()

                for order in orders:

                    # db.refresh(order)

                    awb = order.awb_number
                    courier = order.courier_partner

                    if not awb:
                        continue

                    if courier == "shiprocket" or courier == "Shiprocket":
                        self.track_shipment(awb, db)

            return GenericResponseModel(
                status_code=http.HTTPStatus.OK,
                message="Tracking done",
            )

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error creating shipment: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="Some error occurred",
            )
