import http
from psycopg2 import DatabaseError
import datetime
from sqlalchemy import or_
from sqlalchemy import asc, desc
from sqlalchemy import func
from collections import defaultdict
import pandas as pd
from io import BytesIO
from fastapi.responses import StreamingResponse
import xlsxwriter


from fastapi import Depends
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
import http


from utils.jwt_token_handler import JWTHandler

security = HTTPBearer()


import json

# models
from model.order import Order

# controllers
from controller.context_manager import context_user_data
from controller.context_manager import get_db_session

# service
from service.shipment_service import ShipmentService

# schema
from schema.base import GenericResponseModel
from schema.order import Order_create_request_model, Order_Model, getAllOrders

# utils
from utils.flatten_dict import flatten_dict


from logger import logger


async def get_current_user(
    credentials: HTTPAuthorizationCredentials = Depends(security),
):
    token = credentials.credentials
    payload = JWTHandler.verify_access_token(token)
    return payload


def order_to_dict(order):
    """
    Convert an Order object to a dictionary with all its attributes.
    """
    return {
        column.name: getattr(order, column.name) for column in order.__table__.columns
    }


def drop_timezone(dt):
    return dt.replace(tzinfo=None)


def convert_orders_to_dict(orders):
    return [
        {
            **order_to_dict(order),
            "created_at": drop_timezone(order.created_at),
            "updated_at": drop_timezone(order.updated_at),
        }
        for order in orders
    ]


def convert_to_naive(dt):
    if dt.tzinfo is not None:
        return dt.astimezone(datetime.timezone.utc).replace(tzinfo=None)
    return dt


# Function to convert orders to Excel
def create_excel_from_orders(orders):
    # Create a pandas DataFrame from the list of order dictionaries
    df = pd.DataFrame(orders)

    # Use a BytesIO buffer to store the Excel file in memory
    buffer = BytesIO()
    # Write the DataFrame to the buffer as an Excel file
    with pd.ExcelWriter(buffer, engine="openpyxl") as writer:
        df.to_excel(writer, index=False, sheet_name="Orders")

    # Seek to the beginning of the buffer
    buffer.seek(0)

    return buffer


class OrderService:

    @staticmethod
    def create_order(
        order_data: Order_create_request_model,
    ) -> GenericResponseModel:

        try:

            with get_db_session() as db:

                formatted_order = order_data.model_dump()

                newOrder = {
                    "pickup_name": formatted_order["pickup_details"]["name"],
                    "pickup_phone": formatted_order["pickup_details"]["phone"],
                    "pickup_email": formatted_order["pickup_details"]["email"],
                    "pickup_company": formatted_order["pickup_details"]["company_name"],
                    "pickup_address": formatted_order["pickup_details"]["address"],
                    "pickup_pincode": formatted_order["pickup_details"]["pincode"],
                    "pickup_city": formatted_order["pickup_details"]["city"],
                    "pickup_state": formatted_order["pickup_details"]["state"],
                    "pickup_country": formatted_order["pickup_details"]["country"],
                    "pickup_location_code": formatted_order["pickup_details"][
                        "location_code"
                    ],
                    "consignee_name": formatted_order["consignee_details"]["name"],
                    "consignee_phone": formatted_order["consignee_details"]["phone"],
                    "consignee_email": formatted_order["consignee_details"]["email"],
                    "consignee_company": formatted_order["consignee_details"][
                        "company_name"
                    ],
                    "consignee_address": formatted_order["consignee_details"][
                        "address"
                    ],
                    "consignee_pincode": formatted_order["consignee_details"][
                        "pincode"
                    ],
                    "consignee_city": formatted_order["consignee_details"]["city"],
                    "consignee_state": formatted_order["consignee_details"]["state"],
                    "consignee_country": formatted_order["consignee_details"][
                        "country"
                    ],
                    "billing_name": formatted_order["billing_details"]["name"],
                    "billing_phone": formatted_order["billing_details"]["phone"],
                    "billing_email": formatted_order["billing_details"]["email"],
                    "billing_company": formatted_order["billing_details"][
                        "company_name"
                    ],
                    "billing_address": formatted_order["billing_details"]["address"],
                    "billing_pincode": formatted_order["billing_details"]["pincode"],
                    "billing_city": formatted_order["billing_details"]["city"],
                    "billing_state": formatted_order["billing_details"]["state"],
                    "billing_country": formatted_order["billing_details"]["country"],
                    "company_id": context_user_data.get().company_id,
                    "invoice_amount": formatted_order[""],
                    "invoice_number": "",
                }

                del formatted_order["pickup_details"]
                del formatted_order["billing_details"]
                del formatted_order["consignee_details"]

                api_key = formatted_order["api_key"]
                courier_partner = formatted_order["courier_partner"]

                del formatted_order["api_key"]
                del formatted_order["courier_partner"]

                flattened_order_data = flatten_dict(formatted_order)

                final_order_data = {**newOrder, **flattened_order_data}

                final = Order(**final_order_data)

                order = db.query(Order).filter(Order.order_id == final.order_id).first()

                # check of order already exists for the id and no courier partner is present in the request
                if order and (not courier_partner):
                    return GenericResponseModel(
                        status_code=http.HTTPStatus.CONFLICT,
                        data={"order_id": final.order_id},
                        message="Order Id already exists",
                    )

                # check if shipment already exists for the existing order
                if order and order.awb_number and order.courier_partner != "Shiprocket":

                    if order.courier_partner.lower() == courier_partner.lower():
                        return GenericResponseModel(
                            status_code=http.HTTPStatus.CREATED,
                            data={
                                "awb_number": order.awb_number,
                                "courier_partner": order.courier_partner,
                                "delivery_partner": order.delivery_partner,
                                "label_url": order.label_url,
                                "manifest_url": order.manifest_url,
                                "invoice_url": order.invoice_url,
                            },
                            message="Shipment Already Exists",
                            status=True,
                        )
                    else:
                        return GenericResponseModel(
                            status_code=http.HTTPStatus.CONFLICT,
                            message="Shipment for this Order has already been created",
                            status=False,
                        )

                # check for invalid courier_partner
                if courier_partner:
                    couriers = ["maxlogistics", "shipperfecto", "shiprocket"]

                    if courier_partner not in couriers:
                        return GenericResponseModel(
                            status_code=http.HTTPStatus.UNPROCESSABLE_ENTITY,
                            data={"courier_partner": courier_partner},
                            message="Invalid Courier Partner",
                        )

                # if no existing order is present for the order id, create a new order
                if not order:
                    final.tracking_info = [
                        {
                            "date": datetime.datetime.now().strftime("%d/%m/%Y %H:%M"),
                            "event": "Order Created on Platform",
                        }
                    ]

                    final.number_of_boxes = len(final.package_details)
                    final.source = "Online"
                    final.channel = "Omneelab"

                    db.add(final)
                    db.commit()

                    if not courier_partner:
                        return GenericResponseModel(
                            status_code=http.HTTPStatus.CREATED,
                            data={"order_id": final.order_id},
                            message="Order created Successfully",
                            status=1,
                        )

                if courier_partner:
                    return ShipmentService.create_new_shipment(final, courier_partner)

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error creating Order: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message="An error occurred while creating the Order.",
            )

    @staticmethod
    def get_all_orders(filters: getAllOrders) -> GenericResponseModel:
        try:

            # Retrieve all orders from the database
            with get_db_session() as db:

                filters = filters.model_dump()

                company_id = context_user_data.get().company_id

                page_number = filters["pageNumber"] + 1
                batch_size = filters["batchSize"]

                # Extract status filter
                status_filter = filters["status"]

                search = filters["search"]

                orderType = filters["type"]

                statuses_to_include = [
                    "new",
                    "booked",
                    "delivered",
                    "pickup scheduled",
                    "in transit",
                    "canceled",
                ]
                ndr_statuses_to_include = [
                    "rto delivered",
                    "rto initiated",
                    "rto in intransit",
                    "lost",
                    "cancellation requested",
                    "in transit-en-route",
                    "pickup exception",
                    "rto_ndr",
                ]

                with get_db_session() as db:
                    # Construct base query
                    query = db.query(Order)

                    if company_id:
                        query = query.filter(Order.company_id == company_id)
                    else:
                        query = query.filter(
                            or_(
                                Order.company_id == None,
                                Order.company_id == "",
                            )
                        )

                    if orderType == "forward":
                        # Filter orders where status is in statuses_to_include
                        query = query.filter(
                            func.lower(Order.status).in_(
                                map(str.lower, statuses_to_include)
                            )
                        )
                    else:
                        query = query.filter(
                            func.lower(Order.status).in_(
                                map(str.lower, ndr_statuses_to_include)
                            )
                        )

                    if search:
                        query = query.filter(Order.awb_number.ilike(f"%{search}%"))

                    # Apply status filter if provided
                    if status_filter and status_filter != "All":
                        query = query.filter(Order.status.ilike(status_filter))

                    total_orders = query.count()

                    # Apply pagination
                    offset = (page_number - 1) * batch_size
                    query = (
                        query.order_by(desc(Order.created_at))
                        .offset(offset)
                        .limit(batch_size)
                    )

                    # Execute query to retrieve orders
                    orders = query.all()

            # Convert orders to dictionaries
            orders_dicts = [order_to_dict(order) for order in orders]

            return GenericResponseModel(
                status_code=http.HTTPStatus.OK,
                data={"orders": orders_dicts, "orders_count": total_orders},
                status=True,
                message="All orders retrieved successfully",
            )

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error retrieving orders: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message="An error occurred while retrieving orders.",
            )

        except Exception as e:

            logger.error(
                msg="Error retrieving orders: {}".format(str(e)),
            )

            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message=str(e),
            )

    @staticmethod
    def getExcel(filters: getAllOrders):
        try:
            # Retrieve all orders from the database
            with get_db_session() as db:
                filters = filters.model_dump()

                company_id = context_user_data.get().company_id

                # Extract status filter
                status_filter = filters["status"]

                search = filters["search"]

                orderType = filters["type"]

                statuses_to_include = [
                    "new",
                    "booked",
                    "delivered",
                    "pickup scheduled",
                    "in transit",
                    "canceled",
                ]
                ndr_statuses_to_include = [
                    "rto delivered",
                    "rto initiated",
                    "rto in intransit",
                    "lost",
                    "cancellation requested",
                    "in transit-en-route",
                    "pickup exception",
                    "rto_ndr",
                ]

                with get_db_session() as db:
                    # Construct base query
                    query = db.query(Order)

                    if company_id:
                        query = query.filter(Order.company_id == company_id)
                    else:
                        query = query.filter(
                            or_(
                                Order.company_id == None,
                                Order.company_id == "",
                            )
                        )

                    if orderType == "forward":
                        # Filter orders where status is in statuses_to_include
                        query = query.filter(
                            func.lower(Order.status).in_(
                                map(str.lower, statuses_to_include)
                            )
                        )
                    else:
                        query = query.filter(
                            func.lower(Order.status).in_(
                                map(str.lower, ndr_statuses_to_include)
                            )
                        )

                    if search:
                        query = query.filter(Order.awb_number.ilike(f"%{search}%"))

                    # Apply status filter if provided
                    if status_filter and status_filter != "All":
                        query = query.filter(Order.status.ilike(status_filter))

                    total_orders = query.count()

                    # Apply pagination

                    query = query.order_by(desc(Order.created_at))

                    # Execute query to retrieve orders
                    orders = query.all()

            # Convert orders to dictionaries
            orders_dicts = convert_orders_to_dict(orders)

            # Create Excel file from orders
            excel_buffer = create_excel_from_orders(orders_dicts)

            output = BytesIO()
            workbook = xlsxwriter.Workbook(output)
            worksheet = workbook.add_worksheet()

            worksheet.write(0, 0, "Name")
            worksheet.write(0, 1, "Roll no")
            worksheet.write(0, 2, "Class")
            worksheet.write(0, 3, "section")

            expenses = [
                {"name": "jasvir singh", "rollno": "005", "class": "x", "section": "a"},
                {
                    "name": "lakhvir singh",
                    "rollno": "002",
                    "class": "x",
                    "section": "b",
                },
                {
                    "name": "pardeep singh",
                    "rollno": "003",
                    "class": "x11",
                    "section": "c",
                },
                {
                    "name": "sandeep singh",
                    "rollno": "004",
                    "class": "x111",
                    "section": "d",
                },
            ]

            row = 0
            col = 0
            for key, item in enumerate(expenses):
                incre = key + 1
                worksheet.write(incre, 0, item["name"])
                worksheet.write(incre, 1, item["rollno"])
                worksheet.write(incre, 2, item["class"])
                worksheet.write(incre, 3, item["section"])

            workbook.close()

            headers = {"Content-Disposition": 'attachment; filename="orders.xlsx"'}

            output.seek(0)

            print(output)
            print(worksheet)

            return StreamingResponse(
                output,
                media_type="application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
                headers=headers,
            )

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error retrieving orders: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message="An error occurred while retrieving orders.",
            )

        except Exception as e:
            logger.error(
                msg="Error retrieving orders: {}".format(str(e)),
            )

            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message=str(e),
            )

    @staticmethod
    def get_order_by_id(order_id: str) -> GenericResponseModel:
        try:
            order = []
            with get_db_session() as db:
                order = db.query(Order).filter(Order.order_id == order_id).first()

            # Convert orders to dictionaries
            orders_dicts = order_to_dict(order)

            return GenericResponseModel(
                status_code=http.HTTPStatus.OK,
                data=orders_dicts,
                message="All orders retrieved successfully",
            )

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error retrieving orders: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message="An error occurred while retrieving orders.",
            )

    @staticmethod
    def get_dashboard() -> GenericResponseModel:
        try:

            now = datetime.datetime.now()
            formatted_date = now.strftime("%d-%m-%Y")

            company_id = context_user_data.get().company_id

            with get_db_session() as db:

                orders = []

                if company_id:
                    orders = (
                        db.query(Order).filter(Order.company_id == company_id).all()
                    )
                else:
                    orders = (
                        db.query(Order)
                        .filter(
                            or_(
                                Order.company_id == None,
                                Order.company_id == "",
                            )
                        )
                        .all()
                    )

                # Total number of orders
                total_orders = len(orders)

                print("total orders", total_orders)

                # Number of shipments (orders with a courier_partner)
                number_of_shipments = sum(
                    1 for order in orders if order.courier_partner
                )

                # Average shipment weight and value calculations
                total_weight = 0
                total_value = 0
                shipment_count = 0

                city_data = defaultdict(
                    lambda: {"orders": 0, "total_weight": 0, "total_value": 0}
                )

                client_data = defaultdict(
                    lambda: {
                        "orders": 0,
                        "total_weight": 0,
                        "total_value": 0,
                        "shipments": 0,
                    }
                )

                # Status-wise order count
                status_counts = defaultdict(int)
                ndr_status_counts = defaultdict(int)
                statuses_to_include = [
                    "new",
                    "booked",
                    "delivered",
                    "pickup scheduled",
                    "in transit",
                    "canceled",
                ]
                ndr_statuses_to_include = [
                    "rto delivered",
                    "rto initiated",
                    "rto in intransit",
                    "lost",
                    "cancellation requested",
                    "in transit-en-route",
                    "pickup exception",
                    "rto_ndr",
                ]

                for order in orders:

                    status = order.status.lower()
                    if status in statuses_to_include:
                        status_counts[status] += 1

                    if status in ndr_statuses_to_include:
                        ndr_status_counts[status] += 1

                    total_value += float(
                        order.invoice_amount if order.invoice_amount else 0
                    )

                    if order.courier_partner:
                        shipment_count += 1

                    # Assuming `package_details` is a JSON field or similar
                    if order.package_details and len(order.package_details) > 0:
                        total_weight += float(order.package_details[0]["weight"])

                    city = order.consignee_city

                    if city:  # Ensure city is not None or empty
                        city_data[city]["orders"] += 1
                        city_data[city]["total_weight"] += float(
                            order.package_details[0]["weight"]
                        )
                        city_data[city]["total_value"] += float(
                            order.invoice_amount if order.invoice_amount else 0
                        )

                    client = order.consignee_company

                    if client:  # Ensure city is not None or empty
                        client_data[client]["orders"] += 1
                        if order.courier_partner:
                            client_data[client]["shipments"] += 1
                        client_data[client]["total_weight"] += float(
                            order.package_details[0]["weight"]
                        )
                        client_data[client]["total_value"] += float(
                            order.invoice_amount if order.invoice_amount else 0
                        )

                average_shipment_weight = (
                    total_weight / total_orders if total_orders > 0 else 0
                )
                average_shipment_value = (
                    total_value / total_orders if total_orders > 0 else 0
                )

                top_cities = sorted(
                    city_data.items(), key=lambda x: x[1]["orders"], reverse=True
                )[:10]

                top_clients = sorted(
                    client_data.items(), key=lambda x: x[1]["orders"], reverse=True
                )[:10]

                # Format the top city data
                top_city_stats = [
                    {
                        "city": city,
                        "number_of_orders": data["orders"],
                        "total_weight": round(data["total_weight"], 2),
                        "total_value": round(data["total_value"], 2),
                    }
                    for city, data in top_cities
                ]

                stats = {
                    "total_orders": total_orders,
                    "number_of_shipments": number_of_shipments,
                    "average_shipment_weight": round(average_shipment_weight, 2),
                    "average_shipment_value": round(average_shipment_value, 2),
                }

                top_clients_stats = [
                    {
                        "client": client,
                        "total_orders": data["orders"],
                        "number_of_shipments": data["shipments"],
                        "average_shipment_weight": round(
                            round(data["total_weight"], 2) / data["orders"], 2
                        ),
                        "average_shipment_value": round(
                            round(data["total_value"], 2) / data["orders"], 2
                        ),
                    }
                    for client, data in top_clients
                ]

            data = {
                "stats": stats,
                "top_city_stats": top_city_stats,
                "top_clients": top_clients_stats,
                "status_counts": status_counts,
                "ndr_status_counts": ndr_status_counts,
            }

            # Convert orders to dictionaries
            return GenericResponseModel(
                status_code=http.HTTPStatus.OK,
                data=data,
                status=True,
                message="Successfull",
            )

        except DatabaseError as e:
            # Log database error
            logger.error(
                extra=context_user_data.get(),
                msg="Error retrieving dashbooard: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message="An error occurred while retrieving dashbooard.",
            )
