from enum import Enum
from uuid import UUID

from pydantic import BaseModel, validator, EmailStr

from schema.base import DBBaseModel


# Enum for user roles
class UserRole(str, Enum):
    ADMIN = "admin"
    USER = "user"


# Enum for user status
class UserStatus(str, Enum):
    ACTIVE = "active"
    INACTIVE = "inactive"


# User token data
class UserTokenData(BaseModel):
    uuid: str
    role: UserRole
    email: EmailStr
    first_name: str
    last_name: str
    company_id: str


# User model for response
class UserResponseModel(BaseModel):
    first_name: str
    last_name: str
    email: EmailStr
    role: UserRole = UserRole.USER
    company_id: str
    
class UserResponseNoCompany(BaseModel):
    first_name: str
    last_name: str
    email: EmailStr
    role: UserRole = UserRole.USER


# User base model
class UserBaseModel(UserResponseModel):
    status: UserStatus = UserStatus.ACTIVE


# User model for insert
class UserInsertModel(UserBaseModel):

    password: str

    @validator("password")
    def password_validator(cls, password):
        """
        Validates that the password is at least 8 characters long,
        contains at least one uppercase letter, one lowercase letter,
        one number, and one special character.
        """
        special_chars = {
            "!",
            "@",
            "#",
            "$",
            "%",
            "^",
            "&",
            "*",
            "(",
            ")",
            "-",
            "+",
            "=",
        }
        if len(password) < 8:
            raise ValueError("password must be at least 8 characters long")
        if not any(char.isupper() for char in password):
            raise ValueError("password must contain at least one uppercase letter")
        if not any(char.islower() for char in password):
            raise ValueError("password must contain at least one lowercase letter")
        if not any(char.isdigit() for char in password):
            raise ValueError("password must contain at least one number")
        if not any(char in special_chars for char in password):
            raise ValueError("password must contain at least one special character")
        return password

    # create a db object from the insert model
    def create_db_entity(self, password_hash: str):
        from model.user import User

        dict_to_build_db_entity = self.model_dump()
        dict_to_build_db_entity["password_hash"] = password_hash
        dict_to_build_db_entity.pop("password")

        return User(**dict_to_build_db_entity)


# User Model
class UserModel(UserBaseModel, DBBaseModel):
    password_hash: str

    class Config:
        from_attributes = True


# User login model
class UserLoginModel(BaseModel):
    email: EmailStr
    password: str


class TokenType(str, Enum):
    bearer = "bearer"


class UserTokenResponseModel(BaseModel):
    access_token: str
    user_data: UserResponseNoCompany
