import os, sys
import uvicorn
from fastapi import FastAPI, APIRouter, Request, status
from fastapi.middleware.cors import CORSMiddleware
from contextlib import asynccontextmanager

from fastapi import HTTPException

from fastapi.exceptions import RequestValidationError

from pydantic import BaseModel, ValidationError

from utils.helper import handle_validation_error
from fastapi.responses import JSONResponse

from fastapi import Depends
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
import http

from logger import logger

from utils.jwt_token_handler import JWTHandler

security = HTTPBearer()

# from server.auth import authenticate_token

# Get the absolute path of the root folder and add it
# ROOT_DIR = os.path.abspath(os.path.join(os.path.dirname(__file__), ".."))
# sys.path.append(ROOT_DIR)

from controller import (
    status_controller,
    user_controller,
    shipment_controller,
    order_controller,
)
from model.db import DBBase, db_engine
from fastapi.encoders import jsonable_encoder

# automaticall create the tables in DB
DBBase.metadata.create_all(db_engine)

app = FastAPI()


@app.exception_handler(RequestValidationError)
async def validation_exception_handler(request: Request, exc: RequestValidationError):
    return handle_validation_error(exc)


@app.exception_handler(Exception)
async def global_exception_handler(request: Request, exc: Exception):
    # Log the error with the request context
    logger.error(f"Unhandled exception: {exc} | Path: {request.url.path}")

    return JSONResponse(
        status_code=500,
        content={
            "status_code": 500,
            "message": "An internal server error occurred. Please try again later.",
        },
    )


app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)


# Create a nested router with the common prefix
common_router = APIRouter(prefix="/api/v1")

# Create a default router for api landing page
DefaultRouter = APIRouter()


@DefaultRouter.get("/")
async def hello():
    return "Welcome to the Logistics App Service"


# open routes
common_router.include_router(user_controller.user_router)

# authenticated routes
common_router.include_router(shipment_controller.shipment_router)
common_router.include_router(order_controller.order_router)

# routes
app.include_router(DefaultRouter)
app.include_router(status_controller.router)
app.include_router(common_router)


if __name__ == "__main__":
    uvicorn.run("app:main", host="0.0.0.0", port=8000, reload=True)
