import http
import json
import datetime
import requests
from psycopg2 import DatabaseError

from controller.context_manager import context_user_data
from controller.context_manager import get_db_session

from model.order import Order
from schema.base import GenericResponseModel
from schema.shipment import Shipment_create_model

from logger import logger


def concatenate_strings(data):
    if isinstance(data, dict):
        result = ""
        for value in data.values():
            result += concatenate_strings(value)
        return result
    elif isinstance(data, list):
        result = ""
        for item in data:
            result += concatenate_strings(item)
        return result
    elif isinstance(data, str):
        return data
    else:
        return ""


class ShipRocket:

    @staticmethod
    def login():
        api_url = "https://apiv2.shiprocket.in/v1/external/auth/login"
        body = {"email": "crm+19@warehousity.com", "password": "Crm@12345"}

        response = requests.post(api_url, json=body)
        response.raise_for_status()  # Raise an exception for HTTP errors
        response_data = response.json()

    @staticmethod
    def createOrder(shipment_details: Shipment_create_model):

        formatted_shipment_details = shipment_details.model_dump()

        api_url = "https://apiv2.shiprocket.in/v1/external/auth/login"
        body = {"email": "moksh.jaswal1@omneelab.com", "password": "Moksh@Omnee@2024"}

        response = requests.post(api_url, json=body)
        response.raise_for_status()  # Raise an exception for HTTP errors
        response_data = response.json()

        if not response_data["token"]:
            return GenericResponseModel(
                status_code=http.HTTPStatus.BAD_REQUEST,
                error="",
                message="Yaay",
            )

        token = response_data["token"]

        api_url = "https://apiv2.shiprocket.in/v1/external/orders/create/adhoc"

        body = {
            "order_id": shipment_details.order_id,
            "order_date": shipment_details.order_date,
            "pickup_location": "BIL-B2C",
            "comment": "",
            "billing_customer_name": shipment_details.billing_name,
            "billing_last_name": "",
            "billing_address": shipment_details.billing_address[:190],
            "billing_address_2": "",
            "billing_city": shipment_details.billing_city,
            "billing_pincode": shipment_details.billing_pincode,
            "billing_state": shipment_details.billing_state,
            "billing_country": shipment_details.billing_country or "INDIA",
            "billing_email": shipment_details.billing_email,
            "billing_phone": shipment_details.billing_phone,
            "shipping_is_billing": False,
            "shipping_customer_name": shipment_details.consignee_name,
            "shipping_last_name": "",
            "shipping_address": shipment_details.consignee_address,
            "shipping_address_2": "",
            "shipping_city": shipment_details.consignee_city,
            "shipping_pincode": shipment_details.consignee_pincode,
            "shipping_country": shipment_details.consignee_country or "INDIA",
            "shipping_state": shipment_details.consignee_state,
            "shipping_email": shipment_details.consignee_email,
            "shipping_phone": shipment_details.consignee_phone,
            "order_items": [
                {
                    "name": product["sku_code"],
                    "units": str(product["quantity"]),
                    "selling_price": (
                        int(product["unit_price"]) / int(str(product["quantity"]))
                        if int(product["unit_price"]) / int(str(product["quantity"]))
                        > 0.5
                        else 0.5
                    ),
                    "sku": product["name"][:50],
                }
                for product in formatted_shipment_details["products"]
            ],
            "payment_method": (
                "COD" if shipment_details.payment_mode == "COD" else "PREPAID"
            ),
            "shipping_charges": shipment_details.shipping_charges,
            "giftwrap_charges": 0,
            "transaction_charges": 0,
            "total_discount": 0,
            "sub_total": shipment_details.total_amount,
            "length": formatted_shipment_details["package_details"][0]["length"],
            "breadth": formatted_shipment_details["package_details"][0]["breadth"],
            "height": formatted_shipment_details["package_details"][0]["height"],
            "weight": str(
                float(formatted_shipment_details["package_details"][0]["weight"]) / 1000
            ),
        }

        try:

            headers = {
                "Authorization": "Bearer " + token,
                "Content-Type": "application/json",
            }

            # order creation

            response = requests.post(api_url, json=body, headers=headers, verify=False)
            response_data = response.json()

            print(response_data)

            if response_data.get("errors") is not None:
                return GenericResponseModel(
                    status_code=http.HTTPStatus.BAD_REQUEST,
                    message=concatenate_strings(response_data["errors"]),
                )

            print(response_data)

            order_id = response_data.get("order_id", "")
            shipment_id = response_data.get("shipment_id", "")
            awb_number = response_data.get("awb_code", "")
            courier_name = response_data.get("courier_name", "")

            if not order_id or not shipment_id:
                return GenericResponseModel(
                    status_code=http.HTTPStatus.BAD_REQUEST,
                    message="some error occurred in generating shipment",
                )

            # checking the pincode serviceability

            if not awb_number:

                api_url = (
                    "https://apiv2.shiprocket.in/v1/external/courier/serviceability/"
                )

                body = {"order_id": order_id}

                response = requests.get(
                    api_url, json=body, headers=headers, verify=False
                )

                response_data = response.json()

                print(response_data)

                if response_data.get("errors") is not None:
                    return GenericResponseModel(
                        status_code=http.HTTPStatus.BAD_REQUEST,
                        data=response_data["errors"],
                        message=response_data["message"],
                    )

                dsp = response_data.get("data", {})
                dsp = dsp.get("available_courier_companies", "")

                if not dsp or dsp is None or len(dsp) == 0:
                    return GenericResponseModel(
                        status_code=http.HTTPStatus.BAD_REQUEST,
                        message="No Courier avaialable for the given pincodes",
                    )

                dsp = dsp[0]

                if not dsp or dsp is None:
                    return GenericResponseModel(
                        status_code=http.HTTPStatus.BAD_REQUEST,
                        message="No Courier avaialable for the given pincodes",
                    )

                # assing awb numbers

                api_url = "https://apiv2.shiprocket.in/v1/external/courier/assign/awb/"

                body = {"shipment_id": shipment_id}

                response = requests.post(
                    api_url, json=body, headers=headers, verify=False
                )
                response_data = response.json()

                if response_data.get("awb_assign_status") is None:
                    return GenericResponseModel(
                        status_code=http.HTTPStatus.BAD_REQUEST,
                        message=response_data.get(
                            "message", "Some error occurred in assign awb number"
                        ),
                    )

                # Check if "awb_assign_status" is 0
                if response_data["awb_assign_status"] == 0:
                    return GenericResponseModel(
                        status_code=http.HTTPStatus.BAD_REQUEST,
                        message=response_data.get("response", {})
                        .get("data", {})
                        .get("awb_assign_error", "AWB assign error"),
                    )

                awb_number = response_data["response"]["data"]["awb_code"]
                courier_name = response_data["response"]["data"]["courier_name"]

                with get_db_session() as db:
                    order = (
                        db.query(Order)
                        .filter(Order.order_id == shipment_details.order_id)
                        .first()
                    )

                    order.status = "Pickup Scheduled"
                    order.awb_number = awb_number
                    order.courier_partner = "Shiprocket"
                    order.delivery_partner = courier_name

                    new_tracking_info = [
                        {
                            "event": "Shiprocket Order Created",
                            "date": datetime.datetime.now().strftime("%d/%m/%Y %H:%M"),
                        },
                        {
                            "event": "Shiprocket Shipment Created",
                            "subinfo": "deilvery partner - " + courier_name,
                            "date": datetime.datetime.now().strftime("%d/%m/%Y %H:%M"),
                        },
                    ]

                    track = order.tracking_info.copy()
                    track.append(new_tracking_info)
                    order.tracking_info = track

                    db.add(order)
                    db.commit()

            # generate pickup

            api_url = "https://apiv2.shiprocket.in/v1/external/courier/generate/pickup"

            body = {"shipment_id": shipment_id}

            response = requests.post(api_url, json=body, headers=headers, verify=False)
            response_data = response.json()

            # generate label

            api_url = "https://apiv2.shiprocket.in/v1/external/courier/generate/label"

            body = {"shipment_id": [shipment_id]}

            response = requests.post(api_url, json=body, headers=headers, verify=False)
            response_data = response.json()

            label_url = response_data.get("label_url", "")

            # generate invoice

            api_url = "https://apiv2.shiprocket.in/v1/external/orders/print/invoice"

            body = {"ids": [order_id]}

            response = requests.post(api_url, json=body, headers=headers, verify=False)
            response_data = response.json()

            invoice_url = response_data.get("invoice_url", "")

            # generate manifest

            api_url = "https://apiv2.shiprocket.in/v1/external/manifests/generate"

            body = {"shipment_id": shipment_id}

            response = requests.post(api_url, json=body, headers=headers, verify=False)
            response_data = response.json()

            if "status_code" in response_data and response_data["status_code"] == 400:
                api_url = "https://apiv2.shiprocket.in/v1/external/manifests/print"

                body = {"order_ids": [order_id]}

                response = requests.post(
                    api_url, json=body, headers=headers, verify=False
                )
                response_data = response.json()

                manifest_url = response_data.get("manifest_url", "")

            else:
                manifest_url = response_data.get("manifest_url", "")

            with get_db_session() as db:
                order = (
                    db.query(Order)
                    .filter(Order.order_id == shipment_details.order_id)
                    .first()
                )

                order.status = "Pickup Scheduled"
                order.awb_number = awb_number
                order.courier_partner = "Shiprocket"
                order.delivery_partner = courier_name
                order.label_url = label_url
                order.manifest_url = manifest_url
                order.invoice_url = invoice_url

                db.add(order)
                db.commit()

            return GenericResponseModel(
                status_code=http.HTTPStatus.OK,
                data={
                    "awb_number": awb_number,
                    "label_url": label_url,
                    "manifest_url": manifest_url,
                    "invoice_url": invoice_url,
                    "delivery_partner": courier_name,
                    "shipping_id": str(shipment_id),
                },
                message="Shipment created successfully",
                status=True,
            )

        except DatabaseError as e:
            logger.error(
                extra=context_user_data.get(),
                msg="Error creating shipment: {}".format(str(e)),
            )

            # Return error response
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="An error occurred while creating the shipment.",
            )

        except Exception as e:
            return GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                message=str(e),
            )

    @staticmethod
    def trackShipment(awb_number: str):

        api_url = (
            "https://apiv2.shiprocket.in/v1/external/courier/track/awb/" + awb_number
        )

        response = requests.post(api_url, verify=False)
