import http
from fastapi import APIRouter, Depends
from typing import Dict

# controllers
from controller.context_manager import build_request_context

# schema
from schema.base import GenericResponseModel
from schema.order import Order_create_request_model, order_id_model, getAllOrders

# utils
from utils.helper import build_api_response

# services
from service.order_service import OrderService

from fastapi import Depends
from fastapi.security import HTTPBearer, HTTPAuthorizationCredentials
import http


from utils.jwt_token_handler import JWTHandler

security = HTTPBearer()

# Creating the router for orders
order_router = APIRouter(prefix="/order", tags=["orders"])


async def get_current_user(
    credentials: HTTPAuthorizationCredentials = Depends(security),
):
    token = credentials.credentials
    payload = JWTHandler.decode_access_token(token)
    return payload


# create a new order
@order_router.post(
    "/create",
    dependencies=[Depends(get_current_user)],
    status_code=http.HTTPStatus.CREATED,
    response_model=GenericResponseModel,
)
async def create_order(
    order_data: Order_create_request_model, _=Depends(build_request_context)
):
    try:
        response: GenericResponseModel = OrderService.create_order(
            order_data=order_data
        )
        return build_api_response(response)

    except Exception as e:
        return build_api_response(
            GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="An error occurred while creating the order.",
            )
        )


# get all the orders for a user
@order_router.post(
    "/get_all",
    dependencies=[Depends(get_current_user)],
    status_code=http.HTTPStatus.OK,
    response_model=GenericResponseModel,
)
async def get_all_user_orders(filters: getAllOrders, _=Depends(build_request_context)):
    try:
        response: GenericResponseModel = OrderService.get_all_orders(filters=filters)
        return build_api_response(response)

    except Exception as e:
        return build_api_response(
            GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="An error occurred while fetching orders.",
            )
        )


# get all the orders for a user
@order_router.post(
    "/getOrderExcel",
    dependencies=[Depends(get_current_user)],
    status_code=http.HTTPStatus.OK,
)
async def get_all_user_orders(filters: getAllOrders, _=Depends(build_request_context)):
    try:
        response = OrderService.getExcel(filters=filters)
        return response

    except Exception as e:
        return build_api_response(
            GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="An error occurred while fetching orders.",
            )
        )


# get all the orders for a user
@order_router.get(
    "/get_dashboard_data",
    dependencies=[Depends(get_current_user)],
    status_code=http.HTTPStatus.OK,
    response_model=GenericResponseModel,
)
async def get_dashboard_data(_=Depends(build_request_context)):
    try:
        response: GenericResponseModel = OrderService.get_dashboard()
        return build_api_response(response)

    except Exception as e:
        return build_api_response(
            GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="An error occurred while fetching orders.",
            )
        )


@order_router.post(
    "/",
    status_code=http.HTTPStatus.ACCEPTED,
    response_model=GenericResponseModel,
)
async def get_order_by_id(order_data: order_id_model, _=Depends(build_request_context)):
    try:
        response: GenericResponseModel = OrderService.get_order_by_id(
            order_data.order_id
        )
        return build_api_response(response)

    except Exception as e:
        return build_api_response(
            GenericResponseModel(
                status_code=http.HTTPStatus.INTERNAL_SERVER_ERROR,
                data=str(e),
                message="An error occurred while fetching orders.",
            )
        )
